﻿using System;
using System.Windows.Forms;

using Peak.Can.IsoTp;

namespace PcanIsoTpExample
{
    /// <summary>
    /// Form class handling configuration of an ISO-TP mapping.
    /// </summary>
    public partial class FormMapping : Form
    {
        /// <summary>
        /// The configured mapping.
        /// </summary>
        public MappingStatus m_mapping;

        /// <summary>
        /// Default constructor
        /// </summary>
        public FormMapping()
        {
            InitializeComponent();

            m_mapping = new MappingStatus();
                        
            // Fills target type combobox.
            comboBoxTargetType.Items.Add(new ComboBoxItem(
                "PCANTP_ADDRESSING_PHYSICAL",
                TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL));
            comboBoxTargetType.Items.Add(new ComboBoxItem(
                "PCANTP_ADDRESSING_FUNCTIONAL",
                TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL));
            // Fills message type combobox.
            fillComboBoxMsgType(m_mapping.m_canIdType);
            // Fills format type combobox.
            fillComboBoxFormatType(m_mapping.m_canIdType, m_mapping.m_msgType);

            buttonOk.Enabled = checkMapping();
        }

        /// <summary>
        /// Initializes the components with a pre-defined ISO-TP mapping
        /// </summary>
        /// <param name="mapping">The mapping used to initialize UI components (null for default settigns).</param>
        public void initializeMapping(MappingStatus mapping = null)
        {
            int i;
            ComboBoxItem cbItem;

            if (mapping == null)
                mapping = new MappingStatus();

            // select CAN ID Type in corresponding combobox
            switch (mapping.m_canIdType)
            {
                case TPCANTPIdType.PCANTP_ID_CAN_11BIT:
                default:
                    radioButtonCanId11b.Checked = true;
                    radioButtonCanId29b.Checked = false;
                    break;
                case TPCANTPIdType.PCANTP_ID_CAN_29BIT:
                    radioButtonCanId11b.Checked = false;
                    radioButtonCanId29b.Checked = true;
                    break;
            }
            // select message type in corresponding combobox
            for (i = 0; i < comboBoxMsgType.Items.Count; i++)
            {
                cbItem = (ComboBoxItem)comboBoxMsgType.Items[i];
                if ((TPCANTPMessageType)cbItem.Data == mapping.m_msgType)
                {
                    comboBoxMsgType.SelectedIndex = i;
                    break;
                }
            }
            // select format type in corresponding combobox
            for (i = 0; i < comboBoxFormat.Items.Count; i++)
            {
                cbItem = (ComboBoxItem)comboBoxFormat.Items[i];
                if ((TPCANTPFormatType)cbItem.Data == mapping.m_formatType)
                {
                    comboBoxFormat.SelectedIndex = i;
                    break;
                }
            }
            // select target type in corresponding combobox
            for (i = 0; i < comboBoxTargetType.Items.Count; i++)
            {
                cbItem = (ComboBoxItem)comboBoxTargetType.Items[i];
                if ((TPCANTPAddressingType)cbItem.Data == mapping.m_targetType)
                {
                    comboBoxTargetType.SelectedIndex = i;
                    break;
                }
            }
            // set CAN ID & CAN ID response
            numericUpDownCanId.Value = mapping.m_canId;
            numericUpDownCanIdResp.Value = mapping.m_canIdResponse;
            // set addresses
            numericUpDownSourceAddr.Value = mapping.m_sourceAddr;
            numericUpDownTargetAddr.Value = mapping.m_targetAddr;
            numericUpDownRemoteAddr.Value = mapping.m_remoteAddr;
            // update UI components based on selected values
            updateUi();
        }
        
        /// <summary>
        /// Fills the comboBox "Message Type" based on pre-selected inputs.
        /// </summary>
        /// <param name="idType">CAN ID Type of the mapping.</param>
        private void fillComboBoxMsgType(TPCANTPIdType idType)
        {
            ComboBoxItem cbItem = null;

            // Saves selection if any.
            if (comboBoxFormat.SelectedItem == null)
                cbItem = null;
            else
                cbItem = (ComboBoxItem)comboBoxFormat.SelectedItem;
            comboBoxFormat.Items.Clear();

            // Fills message type combobox.
            comboBoxMsgType.Items.Add(new ComboBoxItem(
                "PCANTP_MESSAGE_DIAGNOSTIC",
                TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC));
            if (idType == TPCANTPIdType.PCANTP_ID_CAN_11BIT)
            {
                comboBoxMsgType.Items.Add(new ComboBoxItem(
                    "PCANTP_MESSAGE_REMOTE_DIAGNOSTIC",
                    TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC));
            }
            // Reminder: PCANTP_FORMAT_ENHANCED (29bits only) does NOT require mappings.
            comboBoxFormat.SelectedItem = cbItem;
        }

        /// <summary>
        /// Fills the comboBox "Format Type" based on pre-selected inputs.
        /// </summary>
        /// <param name="idType">CAN ID Type of the mapping.</param>
        /// <param name="msgType">Message Type of the mapping.</param>
        private void fillComboBoxFormatType(TPCANTPIdType idType, TPCANTPMessageType msgType)
        {
            ComboBoxItem cbItem = null;

            // Saves selection if any.
            if (comboBoxFormat.SelectedItem == null)
                cbItem = null;
            else
                cbItem = (ComboBoxItem)comboBoxFormat.SelectedItem;
            comboBoxFormat.Items.Clear();

            // Fills format type combobox.
            switch (msgType)
            {
                case TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC:
                default:
                    comboBoxFormat.Items.Add(new ComboBoxItem(
                        "PCANTP_FORMAT_NORMAL",
                        TPCANTPFormatType.PCANTP_FORMAT_NORMAL));
                    comboBoxFormat.Items.Add(new ComboBoxItem(
                        "PCANTP_FORMAT_EXTENDED",
                        TPCANTPFormatType.PCANTP_FORMAT_EXTENDED));
                    // Reminder: PCANTP_FORMAT_FIXED_NORMAL does NOT require mappings.
                    break;
                case TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC:
                    // Reminder: PCANTP_FORMAT_MIXED does NOT require mappings with 29 bits CAN ID.
                    if (idType == TPCANTPIdType.PCANTP_ID_CAN_11BIT)
                    {
                        comboBoxFormat.Items.Add(new ComboBoxItem(
                        "PCANTP_FORMAT_MIXED",
                        TPCANTPFormatType.PCANTP_FORMAT_MIXED));
                    }
                    break;
            }
            // Reminder: PCANTP_FORMAT_ENHANCED (29bits only) does NOT require mappings.
            comboBoxFormat.SelectedItem = cbItem;
        }

        /// <summary>
        /// Checks consistency of components selection.
        /// </summary>
        private void updateUi()
        {
            // Note: we do not update the member "Enabled" of UI components directly
            // to prevent flickering.
            bool radioButtonCanId11b_Enabled = true;
            bool radioButtonCanId29b_Enabled = true;
            bool comboBoxMsgType_Enabled = false;
            bool comboBoxFormat_Enabled = false;
            bool comboBoxTargetType_Enabled = false;
            bool numericUpDownCanId_Enabled = false;
            bool numericUpDownCanIdResp_Enabled = false;
            bool numericUpDownSourceAddr_Enabled = false;
            bool numericUpDownTargetAddr_Enabled = false;
            bool numericUpDownRemoteAddr_Enabled = false;

            // ID type must be set to continue
            if (!radioButtonCanId11b.Checked && !radioButtonCanId29b.Checked)
                return;
            comboBoxMsgType_Enabled = true;
            // Message type must be set to continue
            if (getSelectedMsgType() != TPCANTPMessageType.PCANTP_MESSAGE_UNKNOWN)
            {
                switch (getSelectedMsgType())
                {
                    case TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC:
                        numericUpDownRemoteAddr_Enabled = true;
                        break;
                    case TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC:
                        // do NOT enable remote address
                        numericUpDownRemoteAddr.Value = 0x00;
                        break;
                }
                comboBoxFormat_Enabled = true;
                // Format type must be set to continue
                if (getSelectedFormatType() != TPCANTPFormatType.PCANTP_FORMAT_UNKNOWN)
                {
                    comboBoxTargetType_Enabled = true;
                    // Target type must be set to continue
                    if (getSelectedTargetType() != TPCANTPAddressingType.PCANTP_ADDRESSING_UNKNOWN)
                    {
                        switch (getSelectedTargetType())
                        {
                            case TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL:
                                // do NOT enable CAN ID response
                                break;
                            case TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL:
                                numericUpDownCanIdResp_Enabled = true;
                                break;
                        }
                        numericUpDownCanId_Enabled = true;
                        numericUpDownSourceAddr_Enabled = true;
                        numericUpDownTargetAddr_Enabled = true;
                    }
                }
            }
            // Updates components
            this.SuspendLayout();
            radioButtonCanId11b.Enabled = radioButtonCanId11b_Enabled;
            radioButtonCanId29b.Enabled = radioButtonCanId29b_Enabled;
            comboBoxMsgType.Enabled = comboBoxMsgType_Enabled;
            comboBoxFormat.Enabled = comboBoxFormat_Enabled;
            comboBoxTargetType.Enabled = comboBoxTargetType_Enabled;
            numericUpDownCanId.Enabled = numericUpDownCanId_Enabled;
            numericUpDownCanIdResp.Enabled = numericUpDownCanIdResp_Enabled;
            numericUpDownSourceAddr.Enabled = numericUpDownSourceAddr_Enabled;
            numericUpDownTargetAddr.Enabled = numericUpDownTargetAddr_Enabled;
            numericUpDownRemoteAddr.Enabled = numericUpDownRemoteAddr_Enabled;
            buttonOk.Enabled = checkMapping();
            this.ResumeLayout();
        }

        /// <summary>
        /// Checks that all the values for the mapping are filled in.
        /// </summary>
        /// <returns>True if the values are set, false otherwise</returns>
        private bool checkMapping()
        {
            uint canId;
            uint canIdResp;
            TPCANTPIdType canIdType;
            TPCANTPFormatType format;
            TPCANTPMessageType msgType;
            TPCANTPAddressingType targetType;
            byte sourceAddr;
            byte targetAddr;
            byte remoteAddr;

            // get mapping values
            try
            {
                canId = Convert.ToUInt32(numericUpDownCanId.Value);
                canIdResp = Convert.ToUInt32(numericUpDownCanIdResp.Value);
                canIdType = getSelectedIdType();
                msgType = getSelectedMsgType();
                format = getSelectedFormatType();
                targetType = getSelectedTargetType();
                sourceAddr = Convert.ToByte(numericUpDownSourceAddr.Value);
                targetAddr = Convert.ToByte(numericUpDownTargetAddr.Value);
                remoteAddr = Convert.ToByte(numericUpDownRemoteAddr.Value);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, PcanIsoTpExample.Properties.Resources.AppName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }
            // check comboboxes selection
            if (msgType == TPCANTPMessageType.PCANTP_MESSAGE_UNKNOWN)
                return false;
            if (format == TPCANTPFormatType.PCANTP_FORMAT_UNKNOWN)
                return false;
            if (targetType == TPCANTPAddressingType.PCANTP_ADDRESSING_UNKNOWN)
                return false;

            // Functional addressing is a one way communication,
            // there is no need to check the target 
            if (targetType != TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL)
            {
                // source and target addresses should not be the same
                // thus it is the same for the CAN IDs 
                if (sourceAddr == targetAddr || canId == canIdResp)
                    return false;
            }
            return true;
        }

        #region Event Handlers
        /// <summary>
        /// Event handler called when button Ok is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonOk_Click(object sender, EventArgs e)
        {
            // Gets mapping values.
            try
            {
                m_mapping.m_canId = Convert.ToUInt32(numericUpDownCanId.Value);
                m_mapping.m_canIdResponse = Convert.ToUInt32(numericUpDownCanIdResp.Value);
                m_mapping.m_canIdType = getSelectedIdType();
                m_mapping.m_msgType = getSelectedMsgType();
                m_mapping.m_formatType = getSelectedFormatType();
                m_mapping.m_targetType = getSelectedTargetType();
                m_mapping.m_sourceAddr = Convert.ToByte(numericUpDownSourceAddr.Value);
                m_mapping.m_targetAddr = Convert.ToByte(numericUpDownTargetAddr.Value);
                m_mapping.m_remoteAddr = Convert.ToByte(numericUpDownRemoteAddr.Value);
            }
            catch (Exception ex)
            {
                // An error occurs, report problem and aborts Dialog closure.
                MessageBox.Show(ex.Message, PcanIsoTpExample.Properties.Resources.AppName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                this.DialogResult = DialogResult.None;
            }
        }

        /// <summary>
        /// Event handler called when selectin in comboBox "Format Type" is changed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void comboBoxFormat_SelectedIndexChanged(object sender, EventArgs e)
        {
            // Unless Format Mixed is selected, Remote Address is disabled.
            if (getSelectedFormatType() == TPCANTPFormatType.PCANTP_FORMAT_MIXED)
            {
                numericUpDownRemoteAddr.Enabled = true;
            }
            else
            {
                numericUpDownRemoteAddr.Enabled = false;
                numericUpDownRemoteAddr.Value = 0x00;
            }
            // Updates OK button status.
            uiComponent_Leave(this, EventArgs.Empty);
        }
        /// <summary>
        /// Event handler called when selectin in comboBox "Message Type" is changed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void comboBoxMsgType_SelectedIndexChanged(object sender, EventArgs e)
        {
            // Updates comboBox "Format Addressing Type" contents:
            //  available format type is dependant of the can id type and the message type.
            fillComboBoxFormatType(getSelectedIdType(), getSelectedMsgType());
            // Updates OK button status.
            uiComponent_Leave(this, EventArgs.Empty);
        }
        /// <summary>
        /// Event handler called when selectin in comboBox "Target Addressing Type" is changed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void comboBoxTargetType_SelectedIndexChanged(object sender, EventArgs e)
        {
            // Can Id Response is not used if Functional Addressing is selected.
            if (getSelectedTargetType() == TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL)
            {
                numericUpDownCanIdResp.Maximum = CanTpApi.CAN_ID_NO_MAPPING;
                numericUpDownCanIdResp.Value = CanTpApi.CAN_ID_NO_MAPPING;
                numericUpDownCanIdResp.Enabled = false;                
            }
            else
            {
                numericUpDownCanIdResp.Maximum = numericUpDownCanId.Maximum;
                numericUpDownCanIdResp.Enabled = true;
                numericUpDownTargetAddr.Enabled = true;
            }
            // Updates OK button status.
            uiComponent_Leave(this, EventArgs.Empty);
        }
        
        /// <summary>
        /// Event handler called when value is changed in various numericUpDown components.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void numericUpDownComponents_ValueChanged(object sender, EventArgs e)
        {
            // Updates OK button status.
            uiComponent_Leave(this, EventArgs.Empty);
        }
        /// <summary>
        /// Event handler called when radioButton "CAN ID Type" checked value changed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        /// 
        private void radioButtonCanId_CheckedChanged(object sender, EventArgs e)
        {
            // Updates maximum CAN ID values.
            if (radioButtonCanId11b.Checked)
                numericUpDownCanId.Maximum = 0x7FF;
            else
                numericUpDownCanId.Maximum = 0x1FFFFFFF;
            numericUpDownCanIdResp.Maximum = numericUpDownCanId.Maximum;
            // Updates comboBox "Format Addressing Type" contents:
            //  available format type is dependant of the can id type and the message type
            fillComboBoxFormatType(getSelectedIdType(), getSelectedMsgType());
            // Updates OK button status.
            uiComponent_Leave(this, EventArgs.Empty);
        }

        /// <summary>
        /// Event handler called when focus is lost on various components.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void uiComponent_Leave(object sender, EventArgs e)
        {
            // Checks integrity of selected UI values.
            updateUi();
        }
        #endregion

        #region UI Getters
        /// <summary>
        /// Returns the selected CAN ID Type.
        /// </summary>
        /// <returns>The selected CAN ID Type (PCANTP_ID_CAN_11BIT if none is selected).</returns>
        TPCANTPIdType getSelectedIdType()
        {
            if (radioButtonCanId29b.Checked)
                return TPCANTPIdType.PCANTP_ID_CAN_29BIT;
            return TPCANTPIdType.PCANTP_ID_CAN_11BIT;
        }
        /// <summary>
        /// Returns the selected ISO-TP Message Type.
        /// </summary>
        /// <returns>The selected ISO-TP Message Type (PCANTP_MESSAGE_UNKNOWN if none is selected).</returns>
        TPCANTPMessageType getSelectedMsgType()
        {
            if (comboBoxMsgType.SelectedItem == null)
                return TPCANTPMessageType.PCANTP_MESSAGE_UNKNOWN;
            return (TPCANTPMessageType)((ComboBoxItem)comboBoxMsgType.SelectedItem).Data;
        }
        /// <summary>
        /// Returns the selected ISO-TP Format Type.
        /// </summary>
        /// <returns>The selected ISO-TP Format Type (PCANTP_FORMAT_UNKNOWN if none is selected).</returns>
        TPCANTPFormatType getSelectedFormatType()
        {
            if (comboBoxFormat.SelectedItem == null)
                return TPCANTPFormatType.PCANTP_FORMAT_UNKNOWN;
            return (TPCANTPFormatType)((ComboBoxItem)comboBoxFormat.SelectedItem).Data;
        }
        /// <summary>
        /// Returns the selected ISO-TP Target Addressing Type.
        /// </summary>
        /// <returns>The selected ISO-TP Target Addressing Type (PCANTP_ADDRESSING_UNKNOWN if none is selected).</returns>
        TPCANTPAddressingType getSelectedTargetType()
        {
            if (comboBoxTargetType.SelectedItem == null)
                return TPCANTPAddressingType.PCANTP_ADDRESSING_UNKNOWN;
            return (TPCANTPAddressingType)((ComboBoxItem)comboBoxTargetType.SelectedItem).Data;
        }
        #endregion
    }
}
